<?php
require_once 'db.php';

// ===============================================================================================
// MENTOR FUNCTIONS
// ===============================================================================================

function getMentorByCode($code) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM mentors WHERE mentor_code = ? AND status = 'active'");
    $stmt->execute([$code]);
    return $stmt->fetch();
}

// ===============================================================================================
// SURVEY SESSION FUNCTIONS
// ===============================================================================================

function createSurveySession($survey_type, $mentor_id, $device_fingerprint) {
    global $pdo;
    $session_token = bin2hex(random_bytes(32));
    
    $stmt = $pdo->prepare("INSERT INTO survey_sessions (session_token, device_fingerprint, survey_type, mentor_id) VALUES (?, ?, ?, ?)");
    $stmt->execute([$session_token, $device_fingerprint, $survey_type, $mentor_id]);
    
    return $session_token;
}

function getSessionByToken($token) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM survey_sessions WHERE session_token = ? AND is_expired = 0");
    $stmt->execute([$token]);
    return $stmt->fetch();
}

function saveQuestionResponse($session_token, $question_num, $response) {
    global $pdo;
    
    $session = getSessionByToken($session_token);
    if (!$session) return false;
    
    $responses = json_decode($session['responses_temp'], true) ?? [];
    $responses[$question_num] = $response;
    
    $stmt = $pdo->prepare("UPDATE survey_sessions SET responses_temp = ?, current_question = ? WHERE session_token = ?");
    $stmt->execute([json_encode($responses), $question_num, $session_token]);
    
    return true;
}

function completeSurvey($session_token, $respondent_data) {
    global $pdo;
    
    $session = getSessionByToken($session_token);
    if (!$session) return false;
    
    // Check if device already completed
    if (isDeviceCompleted($session['device_fingerprint'])) {
        return false;
    }

    $responses = $session['responses_temp']; // Already JSON string
    $decoded_responses = json_decode($responses, true);
    
    // Calculate Quality Score
    // Assuming time_taken is passed in respondent_data
    $time_taken = $respondent_data['time_taken'] ?? 0;
    $quality_score = calculateQualityScore($decoded_responses, $time_taken, $session['device_fingerprint']);
    $is_bot = isBotSuspected($session['device_fingerprint'], $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT'], $time_taken);
    
    // Insert into survey_responses
    $stmt = $pdo->prepare("INSERT INTO survey_responses (
        survey_type, mentor_id, device_fingerprint, ip_address, user_agent,
        respondent_name, email, phone, responses, 
        completion_percentage, is_completed, time_taken_seconds,
        is_bot_suspected, quality_score, started_at
    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
    
    $stmt->execute([
        $session['survey_type'],
        $session['mentor_id'],
        $session['device_fingerprint'],
        $_SERVER['REMOTE_ADDR'],
        $_SERVER['HTTP_USER_AGENT'],
        $respondent_data['name'] ?? null,
        $respondent_data['email'] ?? null,
        $respondent_data['phone'] ?? null,
        $responses,
        100,
        1,
        $time_taken,
        $is_bot ? 1 : 0,
        $quality_score,
        $session['created_at']
    ]);
    
    $response_id = $pdo->lastInsertId();
    
    // Mark session as expired
    $stmt = $pdo->prepare("UPDATE survey_sessions SET is_expired = 1 WHERE id = ?");
    $stmt->execute([$session['id']]);

    // Update Analytics
    updateAnalyticsMetrics($session['survey_type'], $session['mentor_id'], $time_taken, $is_bot, $quality_score > 0.8, $respondent_data);
    
    return $response_id;
}

function updateRespondentInfo($response_id, $respondent_data) {
    global $pdo;
    
    $stmt = $pdo->prepare("UPDATE survey_responses SET 
        respondent_name = ?, 
        email = ?, 
        phone = ? 
        WHERE id = ?");
        
    return $stmt->execute([
        $respondent_data['name'],
        $respondent_data['email'],
        $respondent_data['phone'],
        $response_id
    ]);
}

// ===============================================================================================
// ANTI-BOT & QUALITY FUNCTIONS
// ===============================================================================================

function isDeviceCompleted($fingerprint) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT id FROM survey_responses WHERE device_fingerprint = ? AND is_completed = 1");
    $stmt->execute([$fingerprint]);
    return $stmt->fetch() ? true : false;
}

function calculateQualityScore($responses, $time_taken, $device_fingerprint) {
    $score = 1.0;
    
    // Rule 1: Too fast (< 120s for 30 questions)
    if ($time_taken < 120) $score -= 0.10;
    
    // Rule 2: Duplicate Device (should be caught earlier, but penalized here if slipped through)
    // Checking if there are other attempts
    global $pdo;
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM survey_responses WHERE device_fingerprint = ?");
    $stmt->execute([$device_fingerprint]);
    if ($stmt->fetchColumn() > 0) $score -= 0.20;
    
    // Rule 3: Pattern detection (Simple check for repeating answers)
    // Assuming responses are like {1: "A", 2: "A", ...}
    if (detectPatternAbuse($responses)) $score -= 0.15;
    
    return max(0, $score);
}

function detectPatternAbuse($responses) {
    if (empty($responses)) return false;
    
    $values = array_values($responses);
    $count = count($values);
    if ($count < 10) return false;
    
    // Check for 20+ consecutive same answers
    $consecutive = 1;
    for ($i = 1; $i < $count; $i++) {
        if ($values[$i] == $values[$i-1]) {
            $consecutive++;
            if ($consecutive >= 20) return true;
        } else {
            $consecutive = 1;
        }
    }
    
    return false;
}

function isBotSuspected($fingerprint, $ip, $user_agent, $time_taken) {
    // 1. Time too fast
    if ($time_taken < 60) return true;
    
    // 2. IP check (simple rate limit check)
    global $pdo;
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM survey_responses WHERE ip_address = ? AND started_at > (NOW() - INTERVAL 1 HOUR)");
    $stmt->execute([$ip]);
    if ($stmt->fetchColumn() > 5) return true;
    
    return false;
}

// ===============================================================================================
// FREEBIE & WHATSAPP FUNCTIONS
// ===============================================================================================

function getFreebieByType($survey_type) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM digital_freebies WHERE freebie_type = ?");
    $stmt->execute([$survey_type]);
    return $stmt->fetch();
}

function sendDigitalFreebie($email, $survey_type) {
    // Placeholder for PHPMailer logic
    // In a real implementation, you'd require PHPMailer here
    return true; 
}

function sendWhatsAppResults($phone, $survey_results) {
    // Placeholder for WhatsApp API
    return true;
}

// ===============================================================================================
// ANALYTICS HELPER
// ===============================================================================================

function updateAnalyticsMetrics($survey_type, $mentor_id, $time_taken, $is_bot, $is_high_quality, $respondent_data) {
    global $pdo;
    $date = date('Y-m-d');
    
    // Check if record exists for today
    $stmt = $pdo->prepare("SELECT id FROM analytics_metrics WHERE metric_date = ? AND survey_type = ? AND mentor_id = ?");
    $stmt->execute([$date, $survey_type, $mentor_id]);
    $metric = $stmt->fetch();
    
    if (!$metric) {
        $stmt = $pdo->prepare("INSERT INTO analytics_metrics (metric_date, survey_type, mentor_id) VALUES (?, ?, ?)");
        $stmt->execute([$date, $survey_type, $mentor_id]);
        $metric_id = $pdo->lastInsertId();
    } else {
        $metric_id = $metric['id'];
    }
    
    // Build update query
    $sql = "UPDATE analytics_metrics SET 
            total_completed = total_completed + 1,
            avg_completion_time_seconds = (avg_completion_time_seconds * (total_completed - 1) + ?) / total_completed";
            
    if ($is_bot) {
        $sql .= ", bot_suspected_count = bot_suspected_count + 1";
    }
    if ($is_high_quality) {
        $sql .= ", high_quality_responses = high_quality_responses + 1";
    }
    if (!empty($respondent_data['email'])) {
         // This is a rough approximation, ideally we track unique emails
    }
    
    $sql .= " WHERE id = ?";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$time_taken, $metric_id]);
}
?>
