<?php
require_once '../includes/auth.php';
requireAdminLogin();

$survey_id = $_GET['id'] ?? null;

if (!$survey_id) {
    header("Location: surveys.php");
    exit;
}

// 1. Fetch Survey Details
$stmt = $pdo->prepare("SELECT * FROM surveys WHERE id = ?");
$stmt->execute([$survey_id]);
$survey = $stmt->fetch();

if (!$survey) {
    die("Survey not found.");
}

// 2. Fetch Questions (to map ID/Sort Order to Text)
// We need to mirror getQuestions logic but maybe get all questions even if survey is inactive?
// Yes, for analytics we want historical context.
$stmt = $pdo->prepare("SELECT id, question_text, sort_order FROM survey_questions WHERE survey_id = ? ORDER BY sort_order ASC");
$stmt->execute([$survey_id]);
$questions_db = $stmt->fetchAll(PDO::FETCH_ASSOC);

$questions = [];
foreach ($questions_db as $q) {
    $stmt_opt = $pdo->prepare("SELECT option_text FROM survey_options WHERE question_id = ? ORDER BY sort_order ASC");
    $stmt_opt->execute([$q['id']]);
    $options = $stmt_opt->fetchAll(PDO::FETCH_COLUMN);
    
    $questions[$q['sort_order']] = [
        'text' => $q['question_text'],
        'options' => $options,
        'counts' => array_fill_keys($options, 0) // Initialize counts
    ];
}

// 3. Fetch Responses
// survey_responses.survey_type corresponds to surveys.slug
$stmt = $pdo->prepare("SELECT responses FROM survey_responses WHERE survey_type = ? AND is_completed = 1");
$stmt->execute([$survey['slug']]);
$responses_db = $stmt->fetchAll(PDO::FETCH_COLUMN);

$total_responses = count($responses_db);

// 4. Process Responses
foreach ($responses_db as $json_response) {
    $response_data = json_decode($json_response, true);
    if (!is_array($response_data)) continue;

    foreach ($response_data as $q_num => $answer) {
        // $q_num corresponds to sort_order
        if (isset($questions[$q_num])) {
            // Handle array answers (if multiple choice allowed later) or single string
            if (is_array($answer)) {
                foreach ($answer as $a) {
                    if (!isset($questions[$q_num]['counts'][$a])) {
                         $questions[$q_num]['counts'][$a] = 0;
                    }
                    $questions[$q_num]['counts'][$a]++;
                }
            } else {
                if (!isset($questions[$q_num]['counts'][$answer])) {
                    $questions[$q_num]['counts'][$answer] = 0;
                }
                $questions[$q_num]['counts'][$answer]++;
            }
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Analytics: <?php echo htmlspecialchars($survey['title']); ?> - BodhiTree Admin</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600&display=swap" rel="stylesheet">
    <style>
        @media print {
            body { background-color: white; }
            .no-print, nav, .sidebar { display: none !important; }
            .content { padding: 0; margin: 0; width: 100%; }
            .shadow { box-shadow: none !important; border: 1px solid #eee; }
            /* Ensure charts are visible */
            canvas { min-height: 300px; width: 100% !important; }
        }
    </style>
</head>
<body class="bg-gray-100 font-sans">
    <div class="flex h-screen overflow-hidden">
        <!-- Sidebar -->
        <div class="bg-slate-800 text-white w-64 flex-shrink-0 sidebar no-print">
            <div class="p-4 text-xl font-bold border-b border-slate-700">BodhiTree Admin</div>
            <nav class="mt-4">
                <a href="dashboard.php" class="block py-3 px-4 hover:bg-slate-700">Dashboard</a>
                <a href="surveys.php" class="block py-3 px-4 bg-slate-700 border-l-4 border-blue-500">Surveys</a>
                <a href="analytics.php" class="block py-3 px-4 hover:bg-slate-700">Analytics</a>
                <a href="mentors.php" class="block py-3 px-4 hover:bg-slate-700">Mentors</a>
                <a href="responses.php" class="block py-3 px-4 hover:bg-slate-700">Responses</a>
                <a href="freebies.php" class="block py-3 px-4 hover:bg-slate-700">Freebies</a>
                <a href="logout.php" class="block py-3 px-4 hover:bg-slate-700 text-red-300">Logout</a>
            </nav>
        </div>

        <!-- Content -->
        <div class="flex-1 overflow-auto p-8 content">
            <div class="flex justify-between items-center mb-8 no-print">
                <div class="flex items-center">
                    <a href="surveys.php" class="text-blue-600 hover:text-blue-800 mr-4">
                        &larr; Back
                    </a>
                    <h1 class="text-3xl font-bold text-gray-800">
                        Analytics: <?php echo htmlspecialchars($survey['title']); ?>
                    </h1>
                </div>
                <button onclick="window.print()" class="bg-blue-600 text-white font-bold py-2 px-4 rounded hover:bg-blue-700 shadow flex items-center transition duration-150">
                    <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5 mr-2" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                    </svg>
                    Download Report
                </button>
            </div>

            <div class="bg-white rounded-lg shadow p-6 mb-8">
                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                    <div class="text-center p-4 bg-blue-50 rounded-lg">
                        <div class="text-gray-500 text-sm uppercase font-semibold">Total Responses</div>
                        <div class="text-4xl font-bold text-blue-600 mt-2"><?php echo $total_responses; ?></div>
                    </div>
                    <div class="text-center p-4 bg-green-50 rounded-lg">
                        <div class="text-gray-500 text-sm uppercase font-semibold">Questions</div>
                        <div class="text-4xl font-bold text-green-600 mt-2"><?php echo count($questions); ?></div>
                    </div>
                    <div class="text-center p-4 bg-purple-50 rounded-lg">
                        <div class="text-gray-500 text-sm uppercase font-semibold">Status</div>
                        <div class="text-4xl font-bold mt-2 <?php echo $survey['is_active'] ? 'text-green-600' : 'text-red-600'; ?>">
                            <?php echo $survey['is_active'] ? 'Active' : 'Inactive'; ?>
                        </div>
                    </div>
                </div>
            </div>

            <div class="space-y-8">
                <?php foreach ($questions as $q_num => $q_data): ?>
                    <div class="bg-white rounded-lg shadow p-6">
                        <h3 class="text-lg font-bold text-gray-800 mb-4">
                            Q<?php echo $q_num; ?>. <?php echo htmlspecialchars($q_data['text']); ?>
                        </h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                            <!-- Table View -->
                            <div>
                                <table class="min-w-full divide-y divide-gray-200">
                                    <thead class="bg-gray-50">
                                        <tr>
                                            <th class="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase">Option</th>
                                            <th class="px-4 py-2 text-right text-xs font-medium text-gray-500 uppercase">Count</th>
                                            <th class="px-4 py-2 text-right text-xs font-medium text-gray-500 uppercase">%</th>
                                        </tr>
                                    </thead>
                                    <tbody class="divide-y divide-gray-200">
                                        <?php 
                                        // Sort by count descending for better readability
                                        arsort($q_data['counts']);
                                        foreach ($q_data['counts'] as $option => $count): 
                                            $percentage = $total_responses > 0 ? round(($count / $total_responses) * 100, 1) : 0;
                                        ?>
                                        <tr>
                                            <td class="px-4 py-2 text-sm text-gray-900"><?php echo htmlspecialchars($option); ?></td>
                                            <td class="px-4 py-2 text-sm text-gray-900 text-right"><?php echo $count; ?></td>
                                            <td class="px-4 py-2 text-sm text-gray-500 text-right"><?php echo $percentage; ?>%</td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <!-- Chart View -->
                            <div>
                                <div class="relative h-64">
                                    <canvas id="chart_<?php echo $q_num; ?>"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <script>
                        new Chart(document.getElementById('chart_<?php echo $q_num; ?>'), {
                            type: 'doughnut',
                            data: {
                                labels: <?php echo json_encode(array_keys($q_data['counts'])); ?>,
                                datasets: [{
                                    data: <?php echo json_encode(array_values($q_data['counts'])); ?>,
                                    backgroundColor: [
                                        '#3B82F6', '#10B981', '#F59E0B', '#EF4444', 
                                        '#8B5CF6', '#EC4899', '#6366F1', '#14B8A6'
                                    ]
                                }]
                            },
                            options: {
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: {
                                    legend: {
                                        position: 'right',
                                        labels: {
                                            boxWidth: 12
                                        }
                                    }
                                }
                            }
                        });
                    </script>
                <?php endforeach; ?>
            </div>
            
            <?php if (empty($questions)): ?>
                <div class="text-center py-12 text-gray-500">
                    No questions found for this survey.
                </div>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
